### A Pluto.jl notebook ###
# v0.20.4

using Markdown
using InteractiveUtils

# ╔═╡ 75c33c02-266b-4b57-a3c4-7d942feef6c4
import Pkg; Pkg.activate(".")

# ╔═╡ d8870852-a4f6-49df-b0df-6255ba6d5820
begin
	using CairoMakie
	using CommonMark
	using CSV
	using DataFrames
	using FFTW
	using Format
	using LaTeXStrings
	using PlutoUI
	using Statistics
end

# ╔═╡ 483145c0-6fa9-415d-a8a7-50b07f87fa6d
md"""
**What is this?**


*This jupyter notebook is part of a collection of notebooks on various topics discussed during the Time Domain Astrophysics course delivered by Stefano Covino at the [Università dell'Insubria](https://www.uninsubria.eu/) in Como (Italy). Please direct questions and suggestions to [stefano.covino@inaf.it](mailto:stefano.covino@inaf.it).*
"""

# ╔═╡ 630f06c6-bc9b-4c17-8c08-e102cb1200d5
md"""
**This is a `Julia` notebook**
"""

# ╔═╡ 085370dc-1828-481e-b844-25459eafec3e
Pkg.instantiate()

# ╔═╡ eec5d097-10bc-4100-beee-d67ddf3eaf85
# ╠═╡ show_logs = false
md"""
$(LocalResource("Pics/TimeDomainBanner.jpg"))
"""

# ╔═╡ ad5b4949-a4c8-4667-8980-a8de6d4b9aef
# ╠═╡ show_logs = false
md"""
# Science Case: Sunspot Number
***

## Nuclear fusion
***

- The energy released by the Sun in a nuclear reaction corresponds to a slight reduction of mass according to Einstein’s equation $E = mc^2$.

- Thermonuclear fusion occurs only at very high temperatures; for example, hydrogen fusion occurs only at temperatures in excess of about $10^7$ K.

- In the Sun, fusion occurs only in the dense, hot core 

$(LocalResource("Pics/sunfusion.jpg"))

- Hydrogen fusion takes place in a core extending from the Sun’s center to about 0.25 solar radius.

- The core is surrounded by a radiative zone extending to about 0.71 solar radius.

- In this zone, energy travels outward through radiative diffusion.

- The radiative zone is surrounded by a rather opaque convective zone of gas at relatively low temperature and pressure. In this zone, energy travels outward primarily through convection.

$(LocalResource("Pics/suninterior.jpg"))

"""

# ╔═╡ 3cedd3d4-eec0-41a9-bad4-961aa52c4186
# ╠═╡ show_logs = false
md"""
### The photosphere
***

- The Sun’s atmosphere has three main layers:
    - the photosphere
    - the chromosphere
    - the corona

- Everything below the solar atmosphere is called the solar interior.

- The visible surface of the Sun, the photosphere, is the lowest layer in the solar atmosphere.

- The spectrum of the photosphere is similar to that of a blackbody at a temperature of 5800 K

$(LocalResource("Pics/sunphotosphere.jpg"))

- Looking at the photospehre it is easy to observe a phenomenon known as *limb darkening*:

$(LocalResource("Pics/limbdarkening.jpg"))

- The external layers of the the Sun are in convective equilibrium. And relatively high-resolution observations show the convective cells:

$(LocalResource("Pics/convectivecells.jpg"))
"""

# ╔═╡ ecdad28d-bdfa-4d04-ac0b-e86f1acbd597
# ╠═╡ show_logs = false
cm"""
## Sunspots
***

$(LocalResource("Pics/sunspots.jpg"))

- Due to magnetic phenomena at the surface, occasionally it is possible to observe sunspots.

- The observations of persistent (for a few weeks) sunspost groups allow one to measure the Sun rotation period (and also the differential rotation).

$(LocalResource("Pics/sunrotation.jpg"))

- It was soon realized that the visibility and number of sunspots follow a ``\sim 11`` years cycle (in reality a ``\sim 22`` magnetic cycle.

$(LocalResource("Pics/sunspotcycle.jpg"))
$(LocalResource("Pics/sunspotpositions.jpg"))

- The Sun’s surface features vary in an 11-year cycle – the sunspot cycle.

- The average number of sunspots increases and decreases in a regular cycle of approximately 11 years, with reversed magnetic polarities from one 11-year cycle to the next.

- This is related to a 22-year cycle (the solar cycle) in which the surface magnetic field increases, decreases, and then increases again with the opposite polarity.

- Two sunspot cycles make up one 22-year solar cycle.

"""

# ╔═╡ 4a05f9f3-35ee-4c3f-86d9-ae90d8229c7a
md"""
### Exercize: Sun's $\sim 11$ year cycle
***
"""

# ╔═╡ 7274e731-f51d-4faf-af93-674c841a999b
begin
	sspot = DataFrame(CSV.File("SN_y_tot_V2.0.txt",missingstring="-1",delim=' ',ignorerepeated=true,header=["Year","Nmean","Nstd","Nobs"]))
	    
	fg1 = Figure()
	
	ax1fg1 = Axis(fg1[1,1],
	    xlabel="Year",
	    ylabel="N",
	    title="Sunspot number historical record"
	    )
	
	lines!(sspot[!,:Year],sspot[!,:Nmean])
	
	fg1
end

# ╔═╡ c9a1a2f7-eab0-42b3-860e-096ee3218408
md"""
- A cyclicity is clealy visible, nevertheless cycles do not repeat identical.

- This implies that theis power at different frequencies, although not necessarily with a strict periodicity.
"""

# ╔═╡ 661310b7-b25e-4973-a9df-7d7cb58bd980
begin
	#We use the FFTW package.
	
	function FourierPeriodogram(signal,fs)
	    N = length(signal)
	    freqs = fftfreq(N,fs)
	    positive = freqs .> 0  
	    ft = fft(signal)
	    powers = abs.(ft).^2
	    return freqs[positive], powers[positive]
	end
	
	ssfreq,ssper = FourierPeriodogram(sspot[!,:Nmean],1)
	
	ssperleahy = 2*ssper/sum(sspot[!,:Nmean])
	
	
	fg2 = Figure()
	
	ax2fg2 = Axis(fg2[1,1],
	    xlabel="Periods (year)",
	    ylabel="Power",
	    title="Sunspot number historical record periodogram"
	    )
	
	lines!(1 ./ssfreq,ssperleahy)
	
	xlims!(0.01,300)
	ylims!(0,5000)
	
	fg2
end

# ╔═╡ c521f09e-d1ca-4437-a002-aa20a03f4002
md"""
- A strong peak (or, as we are going to see, multiple peaks) is present at a $\sim 11$ year period, but there is power at a period of $\sim 60$ years and beyiond.

- Let's see better the "peak" at $\sim 11$ years.
"""

# ╔═╡ 43865de5-a009-4e32-9852-738cc7e4a4ee
begin
	fg3 = Figure()
	
	ax3fg3 = Axis(fg3[1,1],
	    xlabel="Periods (year)",
	    ylabel="Power",
	    title="Sunspot number historical record periodogram"
	    )
	
	lines!(1 ./ssfreq,ssperleahy)
	
	xlims!(2,20)
	ylims!(0,5000)
	
	fg3
end

# ╔═╡ 36bd4afb-7665-414d-8067-8721e52ad28e
begin
	pwmax,idmax = findmax(ssperleahy[2:end])
	printfmtln("Period with maximum power: {:.1f} years", 1 ./ ssfreq[idmax])
end

# ╔═╡ 5c2d7acb-fad3-49ad-8ad0-59c84501763c
md"""
- As a matter of fact, the $\sim 11$ year periodicity is in reality a set of multiple peaks woth periods ranging from approximately 8 to 12 years.
"""

# ╔═╡ 4e40f09e-feb4-46fe-947b-dedb993ce55c
begin
	fg4 = Figure()
	
	ax1fg4 = Axis(fg4[1,1],
	    xlabel=L"Frequency (year$^{-1}$)",
	    ylabel="Power",
	    title="Sunspot number historical record periodogram",
	    yscale = log10
	    )
	
	flt = ssperleahy .> 0
	
	lines!(ssfreq[flt],ssperleahy[flt],label="DFT")
	
	hlines!(2,linestyle=:dash,label="Poisson noise")
	
	axislegend()
	
	fg4
end

# ╔═╡ 9d8f3aed-091f-4cd2-a746-0da33cee6102
md"""
- A logarithmic plot shows a typical red-noise behavior that might be due to different reasons. 

- In any case, this means that for computing the significance of a peak ibe unavoidably need to model the underlying noise.
"""

# ╔═╡ 92ba500c-7a58-4abd-bfc8-4f8ba0ccc043
md"""
## Reference & Material

Material and papers related to the topics discussed in this lecture.

- [Norton et al. (2023) - "Solar Cycle Observations”](https://ui.adsabs.harvard.edu/abs/2023SSRv..219...64N/abstract)
"""

# ╔═╡ 8647ad0a-8f92-46f7-93ce-d7b3d36f7e51
md"""
## Further Material

Papers or sites for examining more closely some of the discussed topics.

- ["Solar cycle progression"](https://www.swpc.noaa.gov/products/solar-cycle-progression)
"""

# ╔═╡ df32f65b-42f6-48c1-9bae-476b17a088f4
md"""
### Credits
***

This notebook contains no external material. 
"""

# ╔═╡ 08202728-5e8e-490d-93f1-722e313cd057
cm"""
## Course Flow

<table>
  <tr>
    <td>Previous lecture</td>
    <td>Next lecture</td>
  </tr>
  <tr>
    <td><a href="./open?path=Lectures/Lecture - Spectral Analysis/Lecture-SpectralAnalysis.jl">Lecture about spectral analysis</a></td>
    <td><a href="./open?path=Lectures/Science Case - X-Ray Binaries/Lecture-X-RayBinaries.jl">Science case about X-ray binaries</a></td>
  </tr>
 </table>


"""

# ╔═╡ b593377d-fb58-49ec-b922-8c967fc94eb8
md"""
**Copyright**

This notebook is provided as [Open Educational Resource](https://en.wikipedia.org/wiki/Open_educational_resources). Feel free to use the notebook for your own purposes. The text is licensed under [Creative Commons Attribution 4.0](https://creativecommons.org/licenses/by/4.0/), the code of the examples, unless obtained from other properly quoted sources, under the [MIT license](https://opensource.org/licenses/MIT). Please attribute the work as follows: *Stefano Covino, Time Domain Astrophysics - Lecture notes featuring computational examples, 2025*.
"""

# ╔═╡ 327939a9-9f3a-43ba-bd55-a3e24da7efd5


# ╔═╡ Cell order:
# ╟─483145c0-6fa9-415d-a8a7-50b07f87fa6d
# ╟─630f06c6-bc9b-4c17-8c08-e102cb1200d5
# ╠═75c33c02-266b-4b57-a3c4-7d942feef6c4
# ╠═085370dc-1828-481e-b844-25459eafec3e
# ╠═d8870852-a4f6-49df-b0df-6255ba6d5820
# ╟─eec5d097-10bc-4100-beee-d67ddf3eaf85
# ╟─ad5b4949-a4c8-4667-8980-a8de6d4b9aef
# ╟─3cedd3d4-eec0-41a9-bad4-961aa52c4186
# ╟─ecdad28d-bdfa-4d04-ac0b-e86f1acbd597
# ╟─4a05f9f3-35ee-4c3f-86d9-ae90d8229c7a
# ╟─7274e731-f51d-4faf-af93-674c841a999b
# ╟─c9a1a2f7-eab0-42b3-860e-096ee3218408
# ╠═661310b7-b25e-4973-a9df-7d7cb58bd980
# ╟─c521f09e-d1ca-4437-a002-aa20a03f4002
# ╠═43865de5-a009-4e32-9852-738cc7e4a4ee
# ╠═36bd4afb-7665-414d-8067-8721e52ad28e
# ╟─5c2d7acb-fad3-49ad-8ad0-59c84501763c
# ╠═4e40f09e-feb4-46fe-947b-dedb993ce55c
# ╟─9d8f3aed-091f-4cd2-a746-0da33cee6102
# ╟─92ba500c-7a58-4abd-bfc8-4f8ba0ccc043
# ╟─8647ad0a-8f92-46f7-93ce-d7b3d36f7e51
# ╟─df32f65b-42f6-48c1-9bae-476b17a088f4
# ╟─08202728-5e8e-490d-93f1-722e313cd057
# ╟─b593377d-fb58-49ec-b922-8c967fc94eb8
# ╠═327939a9-9f3a-43ba-bd55-a3e24da7efd5
